- [API Reference - `object`](#api-reference---object)
  - [Getting object info](#getting-object-info)
    - [`object.get_object_group(object_id)`](#objectget_object_groupobject_id)
    - [`object.get_robot_id(object_id)`](#objectget_robot_idobject_id)
  - [Getting object state](#getting-object-state)
    - [`object.position(object_id)`](#objectpositionobject_id)
    - [`object.gps_position(object_id)`](#objectgps_positionobject_id)
    - [`object.rotation_angles(object_id)`](#objectrotation_anglesobject_id)
    - [`object.rotation_quat(object_id)`](#objectrotation_quatobject_id)
    - [`object.linear_velocity(object_id)`](#objectlinear_velocityobject_id)
    - [`object.angular_velocity(object_id)`](#objectangular_velocityobject_id)
    - [`object.acceleration(object_id)`](#objectaccelerationobject_id)
  - [Setting object state](#setting-object-state)
    - [`object.set_position(object_id, position)`](#objectset_positionobject_id-position)
    - [`object.set_gps_position(object_id, position)`](#objectset_gps_positionobject_id-position)
    - [`object.set_rotation_angles(object_id, rotation)`](#objectset_rotation_anglesobject_id-rotation)
    - [`object.set_rotation_quat(object_id, rotation)`](#objectset_rotation_quatobject_id-rotation)
  - [Controlling vehicles](#controlling-vehicles)
    - [`object.attitude_control(object_id, control)`](#objectattitude_controlobject_id-control)
    - [`object.velocity_control(object_id, control)`](#objectvelocity_controlobject_id-control)
    - [`object.position_control(object_id, control)`](#objectposition_controlobject_id-control)
    - [`object.actuator_control(object_id, control)`](#objectactuator_controlobject_id-control)
  - [Camera images](#camera-images)
    - [`object.request_image(camera_id)`](#objectrequest_imagecamera_id)
  - [Getting bounding volumes](#getting-bounding-volumes)
    - [`object.compute_aabb(object_id)`](#objectcompute_aabbobject_id)
    - [`object.compute_bounding_sphere(object_id)`](#objectcompute_bounding_sphereobject_id)
  - [Despawning objects](#despawning-objects)
    - [`object.despawn(object_id)`](#objectdespawnobject_id)

## API Reference - `object`

This module allows you to:
 - query object state (position, rotation, velocity, angular velocity, acceleration)
 - set object state (position and rotation only)
 - if object is a vehicle, pass commands to its controller (go to position, set velocity, etc.)

All functions in this module require object ID (see `map.list_objects` and `map.list_vehicles`). In lua, they are passed as first argument, in websocket API they are passed as part of the command.

### Getting object info

#### `object.get_object_group(object_id)`

Returns the type of the object.

Currently available groups are: `terrain`, `obstacle`, `vehicle`, `goal`.

```lua
local object_group = object.get_object_group("22v1")
```

```js
let object_group = await client.rpc('object_22v1.get_object_group')
```

#### `object.get_robot_id(object_id)`

Returns the robot ID associated with the vehicle. Returns `null` if the object is not a vehicle or doesn't have robot ID assigned to it.

```lua
local robot_id = object.get_robot_id("33v1")
```

```js
let robot_id = await client.rpc('object_33v1.get_robot_id')
```

### Getting object state

You can return all properties you need in one network call using the following script:

```js
let state = await client.rpc('script.eval', {
  code: `{
    position = sim.object.position(ARGS[1]),
    rotation = sim.object.rotation_angles(ARGS[1]),
    lin_vel = sim.object.linear_velocity(ARGS[1]),
    ang_vel = sim.object.angular_velocity(ARGS[1]),
    accel = sim.object.acceleration(ARGS[1]),
  }`,
  args: ["22v1"],
})
```

#### `object.position(object_id)`

Returns object position in local ENU coordinates.

```lua
local position = object.position("22v1")
```

```js
let position = await client.rpc('object_22v1.position')
// or subscribe
let sub = client.newSubscription('object_22v1.position')
```

Return value is a table with three fields - `x` (f32), `y` (f32), `z` (f32).

#### `object.gps_position(object_id)`

Returns object position in GPS coordinates.

```lua
local gps_position = object.gps_position("22v1")
```

```js
let gps_position = await client.rpc('object_22v1.gps_position')
// or subscribe
let sub = client.newSubscription('object_22v1.gps_position')
```

Return value is a table with three fields - `lat` (f32), `lon` (f32), `alt` (f32).

#### `object.rotation_angles(object_id)`

Returns object rotation in euler angles (roll, pitch, yaw). Return value is a bivector (YZ is roll, ZX is pitch, XY is yaw).

```lua
local rotation = object.rotation_angles("22v1")
```

```js
let rotation = await client.rpc('object_22v1.rotation_angles')
// or subscribe
let sub = client.newSubscription('object_22v1.rotation_angles')
```

#### `object.rotation_quat(object_id)`

Returns object rotation in quaternion form (W, X, Y, Z).

```lua
local rotation = object.rotation_quat("22v1")
```

```js
let rotation = await client.rpc('object_22v1.rotation_quat')
// or subscribe
let sub = client.newSubscription('object_22v1.rotation_quat')
```

#### `object.linear_velocity(object_id)`

Returns object linear velocity in local coordinates, world frame of reference.

```lua
local lin_vel = object.linear_velocity("22v1")
```

```js
let lin_vel = await client.rpc('object_22v1.linear_velocity')
// or subscribe
let sub = client.newSubscription('object_22v1.linear_velocity')
```

#### `object.angular_velocity(object_id)`

Returns object angular velocity in local coordinates, world frame of reference.

```lua
local ang_vel = object.angular_velocity("22v1")
```

```js
let ang_vel = await client.rpc('object_22v1.angular_velocity')
// or subscribe
let sub = client.newSubscription('object_22v1.angular_velocity')
```

#### `object.acceleration(object_id)`

Returns object acceleration in local coordinates, world frame of reference.

```lua
local accel = object.acceleration("22v1")
```

```js
let accel = await client.rpc('object_22v1.acceleration')
// or subscribe
let sub = client.newSubscription('object_22v1.acceleration')
```

### Setting object state

#### `object.set_position(object_id, position)`

Sets object position to a given value in local coordinates.

```lua
object.set_position("22v1", { x = 1, y = 2, z = 3 })
```

```js
await client.rpc('object_22v1.set_position', { x: 1, y: 2, z: 3 })
```

#### `object.set_gps_position(object_id, position)`

Sets object position to a given value in GPS coordinates.

```lua
object.set_gps_position("22v1", { lat = 50.844968, lon = 4.363814, alt = 10.0 })
```

```js
await client.rpc('object_22v1.set_gps_position', { lat: 50.844968, lon: 4.363814, alt: 10.0 })
```

#### `object.set_rotation_angles(object_id, rotation)`

Sets object rotation to a given value in euler angles (roll, pitch, yaw).

```lua
object.set_rotation_angles("22v1", { yz = 0, zx = 0, xy = 0 })
```

```js
await client.rpc('object_22v1.set_rotation_angles', { yz: 0, zx: 0, xy: 0 })
```

#### `object.set_rotation_quat(object_id, rotation)`

Sets object rotation to a given value in quaternion form.

```lua
object.set_rotation_quat("22v1", { w = 1, x = 0, y = 0, z = 0 })
```

```js
await client.rpc('object_22v1.set_rotation_quat', { w: 1, x: 0, y: 0, z: 0 })
```

### Controlling vehicles

#### `object.attitude_control(object_id, control)`

Activate vehicle controller with given parameters for attitude control.

Parameters are a table with the following fields:
 - `z` (f32) - altitude
 - `vxy` (f32) - yaw rate
 - `zx` (f32) - pitch
 - `yz` (f32) - roll

```lua
object.attitude_control("33v1", { z = 1, vxy = 0, zx = 0, yz = 0 })
```

```js
await client.rpc('object_33v1.attitude_control', { z: 1, vxy: 0, zx: 0, yz: 0 })
```

#### `object.velocity_control(object_id, control)`

Activate vehicle controller with given parameters for velocity control.

Parameters are a table with the following fields:
 - `z` (f32) - altitude
 - `vxy` (f32) - yaw rate
 - `vx` (f32) - velocity forward
 - `vy` (f32) - velocity lateral

```lua
object.velocity_control("33v1", { z = 1, vxy = 0, vx = 0, vy = 0 })
```

```js
await client.rpc('object_33v1.velocity_control', { z: 1, vxy: 0, vx: 0, vy: 0 })
```

#### `object.position_control(object_id, control)`

Activate vehicle controller with given parameters for position control.

Parameters are a table with the following fields:
 - `z` (f32) - altitude
 - `x` (f32) - forward
 - `y` (f32) - left
 - `xy` (f32) - yaw

```lua
object.position_control("33v1", { z = 1, x = 0, y = 0, xy = 0 })
```

```js
await client.rpc('object_33v1.position_control', { z: 1, x: 0, y: 0, xy: 0 })
```

#### `object.actuator_control(object_id, control)`

Set actuator values directly instead of using a controller. Control is a list of actuator values (array of f32). Length and meaning of values depend on the vehicle type.

For UAVs, there are 4 actuators, one for each rotor (front-left, front-right, rear-left, rear-right). Values are between 0 and 1.

For UGVs, there are 12 actuators. First 4 are engine forces for each wheel, next 4 are steering, last 4 are brakes.

```lua
object.actuator_control("33v1", { 0.5, 0.5, 0.5, 0.5 })
```

```js
await client.rpc('object_33v1.actuator_control', [0.5, 0.5, 0.5, 0.5])
```

### Camera images

#### `object.request_image(camera_id)`

Request an image from previously created camera.

Please use camera ID returned from `map.spawn_camera` function, not vehicle ID here. Camera is an object, but it is not usually listed in spatial queries.

This function is returns a table with the following fields:
 - `time_us` (u64) - time elapsed since start of simulation in microseconds
 - `data` (string) - image encoded as data URI

```js
let image = await client.rpc('object_44v1.request_image')
console.log(image.data.data) // data:image/png;base64,...
```

There might be a significant delay after image is requested due to rendering and encoding, so timestamp is provided to help you sync images with other data.

When camera is initializing, first few frames received might be black.

Capturing images is not yet available in LUA.

### Getting bounding volumes

#### `object.compute_aabb(object_id)`

Compute axis-aligned bounding box of the given object.

This function returns a table with the following fields:
 - `min` (vec3)
 - `max` (vec3)

```lua
local aabb = object.compute_aabb("22v1")
```

```js
let aabb = await client.rpc('object_22v1.compute_aabb')
```

#### `object.compute_bounding_sphere(object_id)`

Compute bounding sphere of the given object.

This function returns a table with the following fields:
 - `center` (vec3)
 - `radius` (f32)

```lua
local sphere = object.compute_bounding_sphere("22v1")
```

```js
let sphere = await client.rpc('object_22v1.compute_bounding_sphere')
```

### Despawning objects

#### `object.despawn(object_id)`

Remove an object from the map.

All objects are removevd when the simulation restarts, so this method is mainly used to remove objects in the middle of simulation.

```lua
object.despawn("22v1")
```
